<?php


namespace App\Http\Controllers\Foundation;

use App\Models\FoundationDesignation;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\BloodGroup;
use App\Models\District;
use App\Models\Division;
use App\Models\Union;
use App\Models\Upazilla;
use App\Models\Village;
use App\Models\Ward;
use Illuminate\Support\Facades\Hash;

class FoundationDesignationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function getAllData(Request $request)
    {
        $division_id = $request->division_id;

        $districts = District::where('division_id', $division_id)->get();
        $upazillas = Upazilla::where('district_id', $request->district_id)->get();
        $unions = Union::where('upazilla_id', $request->upazilla_id)->get();
        $wards = Ward::where('union_id', $request->union_id)->get();
        $villages = Village::where('ward_id', $request->ward_id)->get();

        return response()->json([
            'districts' => $districts,
            'upazillas' => $upazillas,
            'unions' => $unions,
            'wards' => $wards,
            'villages' => $villages,
        ]);
    }

    public function index(Request $request)
    {
        $foundationDesignationId = $request->query('hospital_id');
        $foundationDesignations = FoundationDesignation::latest()->get();
        $selectedFoundationDesignation = null;
        if ($foundationDesignationId) {
            $selectedFoundationDesignation = FoundationDesignation::find($foundationDesignationId);
        }
        return view('foundation.addDesignation.index', compact('foundationDesignations','selectedFoundationDesignation'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $divisions = Division::all();

        $bloodGroups = BloodGroup::all();
        return view('foundation.addDesignation.create', compact('divisions', 'bloodGroups'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // dd($request->all);
        // Validate the incoming request
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:foundation_designations,email',
            'mobile' => 'nullable|string|max:15',
            'password' => 'required|string|min:6',
            'designation' => 'required|string',
            'gender' => 'required|string',
            'division_id' => 'required|exists:divisions,id',
            'district_id' => 'required|exists:districts,id',
            'upazilla_id' => 'required|exists:upazillas,id',
            'union_id' => 'nullable|exists:unions,id',
            'ward_id' => 'nullable|string',
            'village_id' => 'nullable|exists:villages,id',
            'blood_group' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'nid_no' => 'nullable|string|max:20',
            'status' => 'nullable|boolean',
        ]);

        // Create a new FoundationDesignation instance
        $foundationDesignation = new FoundationDesignation();
        $foundationDesignation->name = $request->name;
        $foundationDesignation->email = $request->email;
        $foundationDesignation->mobile = $request->mobile;
        $foundationDesignation->password = Hash::make($request->password);
        $foundationDesignation->designation = $request->designation;
        $foundationDesignation->gender = $request->gender;
        $foundationDesignation->division_id = $request->division_id;
        $foundationDesignation->district_id = $request->district_id;
        $foundationDesignation->upazilla_id = $request->upazilla_id;
        $foundationDesignation->union_id = $request->union_id;
        $foundationDesignation->ward_id = $request->ward_id;
        $foundationDesignation->village_id = $request->village_id;
        $foundationDesignation->blood_group = $request->blood_group;
        $foundationDesignation->nid_no = $request->nid_no;
        $foundationDesignation->status = $request->has('status') ? 1 : 0;


        if ($request->hasFile('image') && $request->file('image')->isValid()) {
            $image = $request->file('image');
            $imageName = uniqid() . '-' . time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('foundation/designation'), $imageName);
            $foundationDesignation->image = 'foundation/designation/' . $imageName;
        }


        // Save the model
        $foundationDesignation->save();

        // Generate and update account ID
        $prefix = 'FD';
        $generatedId = str_pad($foundationDesignation->id, 9, '0', STR_PAD_LEFT);
        $foundationDesignation->account_id = $prefix . $generatedId;
        $foundationDesignation->save();

        // Redirect back with success message
        return redirect()->route('foundation.addDesignation.index')->with('success', 'Designation added successfully.');
    }


    /**
     * Display the specified resource.
     *
     * @param  \App\Models\FoundationDesignation  $foundationDesignation
     * @return \Illuminate\Http\Response
     */
    public function show(FoundationDesignation $foundationDesignation)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\FoundationDesignation  $foundationDesignation
     * @return \Illuminate\Http\Response
     */
    public function edit(FoundationDesignation $foundationDesignation)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\FoundationDesignation  $foundationDesignation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, FoundationDesignation $foundationDesignation)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\FoundationDesignation  $foundationDesignation
     * @return \Illuminate\Http\Response
     */
    public function destroy(FoundationDesignation $foundationDesignation)
    {
        //
    }
}