<?php

namespace App\Http\Controllers\Auth;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Foundation;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Carbon\Carbon;

class FoundationAuthController extends Controller
{
    public function login()
    {
        return view('foundation.login');
    }

    public function save_login(Request $request)
    {
        if (Auth::guard('foundation')->attempt(['email' => $request->email, 'password' => $request->password])) {
            return redirect()->intended(route('foundation.dashboard'))->with('success', 'Login successfully completed');
        }

        return back()->with('error', 'The provided credentials do not match our records.');
    }

    public function contact()
    {

        return view('foundation.test');
    }

    public function register()
    {
        return view('foundation.register');
    }

    public function save_register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'mobile' => 'nullable|string|max:255',
            'email' => 'required|email|max:255|unique:foundations,email',
            'password' => 'required|max:64'
        ]);

        $input = $request->only(['name', 'mobile', 'email']);
        $input['register_no'] = generateUniqueCode(9);

        if ($request->has('password')) {
            $input['password'] = Hash::make($request->password);
        }

        if ($foundation = Foundation::create($input)) {
            Auth::guard('foundation')->login($foundation);

            // Generate OTP
            $otp = Str::upper(Str::random(6));
            $foundation->last_otp = $otp;
            $foundation->otp_sent_count = 1;
            $foundation->otp_created_at = Carbon::now();

            // Update the OTP fields in the database
            $foundation->save();

            // Redirect to OTP verification page
            return redirect()->route('foundation.verify.otp', ['user_id' => $foundation->id])
                ->with('info', 'Please verify your OTP to complete registration.');
        }

        return back()->with('error', 'Something went wrong. Please try again later.');
    }
    public function verifyOtpFoundation(Request $request)
    {
        // Retrieve the authenticated user
        $foundation = Auth::guard('foundation')->user();

        // Check if the foundation user is authenticated
        if (!$foundation) {
            return redirect()->route('foundation.login')->with('error', 'You need to be logged in to verify OTP.');
        }

        $otpExpiryDuration = 5 * 60; // 5 minutes in seconds

        // Ensure `otp_created_at` is handled as a Carbon instance
        $otpCreatedAt = $foundation->otp_created_at ? Carbon::parse($foundation->otp_created_at) : null;
        $otpExpiresAt = $otpCreatedAt ? $otpCreatedAt->addSeconds($otpExpiryDuration) : null;

        // Pass the OTP expiry time to the view
        return view('foundation.verify', compact('foundation', 'otpExpiresAt'));
    }


    public function verifyOtpMatch(Request $request)
    {
        $foundation = Auth::guard('foundation')->user();

        $otpExpiryTime = 5; // OTP expiry in minutes
        if ($foundation->otp_created_at && $foundation->otp_created_at->diffInMinutes(now()) > $otpExpiryTime) {
            return redirect()->back()->with('error', 'OTP has expired. Please request a new one.');
        }

        // Verify OTP
        if ($request->otp === $foundation->last_otp) {
            $foundation->update([
                'last_otp' => null,
                'otp_created_at' => null,
                'otp_sent_count' => 0,
            ]);

            return redirect()->route('foundation.dashboard')->with('success', 'OTP verified successfully!');
        } else {
            return redirect()->back()->with('error', 'Invalid OTP. Please try again.');
        }
    }
}
